/**
 * External dependencies
 */
 import { isEmpty, values as __values, isEqual, noop, pickBy } from 'lodash';
 import classNames from 'classnames';
 
 /**
  * WordPress dependencies
  */
 import { __ } from '@wordpress/i18n';
 import { Button, Tooltip } from '@wordpress/components';
 import { useState } from '@wordpress/element';
 import { useSelect, useDispatch, select } from '@wordpress/data';
 
 /**
  * Internal dependencies
  */
 import { SVGBlob1, SVGSquare, SVGCircle, IconSquare, IconIndivisual } from './icons';
 
 /**
  * @param  uniqueSelector Css selector.
  * @param  props          block props.
  */
  export const checkDuplicateId = (uniqueSelector, props) => {
    const likeBlocks = document.querySelectorAll(uniqueSelector);
    if (likeBlocks && likeBlocks.length > 1) {
        const {clientId, setAttributes} = props;
        setAttributes({uniqueID: clientId});
    }
};

export const checkAttrVal = (data, screen = 'desktop') => {
    let flag = false;
    if (undefined === data || null === data || '' === data) {
        return flag;
    }
    if ('object' === typeof data && data.hasOwnProperty([screen]) && '' !== data[screen]) {
        flag = true;
    }
    return flag;
}
export const fontCheck = (props) => {
    let fontArray = [];
    fontArray = pickBy(props, function (value, key) {

        return key.match(/font/gi);
    });
    if (fontArray && Object.keys(fontArray).length) {
        Object.keys(fontArray).forEach(font => includeFont(fontArray[font]));
    }
}

export const includeFont = (font) => {
    if (font) {
        const fonts = [
            font.desktop && font.desktop.family && !systemFonts.includes(font.desktop.family)
                ? (
                    font.desktop.family +
                    ':' +
                    (font.desktop.weight ? font.desktop.weight : 'regular')
                ).replace(/\s/g, '+')
                : '',
            font.tablet && font.tablet.family && !systemFonts.includes(font.desktop.family)
                ? (
                    font.tablet.family +
                    ':' +
                    (font.tablet.weight ? font.tablet.weight : 'regular')
                ).replace(/\s/g, '+')
                : '',
            font.mobile && font.mobile.family && !systemFonts.includes(font.desktop.family)
                ? (
                    font.mobile.family +
                    ':' +
                    (font.mobile.weight ? font.mobile.weight : 'regular')
                ).replace(/\s/g, '+')
                : '',
        ].filter(function (el) {
            return el !== '';
        });
        if (fonts.length) {
            WebFont.load({
                google: {
                    families: fonts,
                },
            });
            includeFontInIframe(fonts)
        }
    }
};

/**
 *
 * @param {string} cssAttr created for padding and margin computation
 * @param {Object} data    padding, margin data collection
 * @param {string} screen  responsive screen
 * @return
 */
export const cssSpaceCompute = (cssAttr, data, screen = 'desktop') => {
    if (isEmpty(data[screen])) {
        return false;
    }

    if ('desktop' !== screen) {
        if (isEqual(data[screen], data.desktop)) {
            return false;
        }
    }
    const {unit, ...tempVal} = data[screen];
    const tempArr = __values(tempVal);

    if (tempArr.every((item) => item === tempArr[0])) {
        return `${cssAttr}: ${tempArr[0]}${unit};`;
    }

    let spaceCSS = '';
    if (!isEmpty(tempVal.top)) {
        spaceCSS += `${cssAttr}: ${tempVal.top}${unit};`;
    }
    if (!isEmpty(tempVal.right)) {
        spaceCSS += `${cssAttr}: ${tempVal.right}${unit};`;
    }
    if (!isEmpty(tempVal.bottom)) {
        spaceCSS += `${cssAttr}: ${tempVal.bottom}${unit};`;
    }
    if (!isEmpty(tempVal.left)) {
        spaceCSS += `${cssAttr}: ${tempVal.left}${unit};`;
    }
    return spaceCSS;
};

/**
 * Default colors for color pallete
 */
export const defaultColors = [
    {
        name: __('Black', 'funnel-builder'),
        slug: 'black',
        color: '#000000',
    },
    {
        name: __('Cyan bluish gray', 'funnel-builder'),
        slug: 'cyan-bluish-gray',
        color: '#abb8c3',
    },
    {
        name: __('White', 'funnel-builder'),
        slug: 'white',
        color: '#ffffff',
    },
    {
        name: __('Pale pink', 'funnel-builder'),
        slug: 'pale-pink',
        color: '#f78da7',
    },
    {
        name: __('Vivid red', 'funnel-builder'),
        slug: 'vivid-red',
        color: '#cf2e2e',
    },
    {
        name: __('Luminous vivid orange', 'funnel-builder'),
        slug: 'luminous-vivid-orange',
        color: '#ff6900',
    },
    {
        name: __('Luminous vivid amber', 'funnel-builder'),
        slug: 'luminous-vivid-amber',
        color: '#fcb900',
    },
    {
        name: __('Light green cyan', 'funnel-builder'),
        slug: 'light-green-cyan',
        color: '#7bdcb5',
    },
    {
        name: __('Vivid green cyan', 'funnel-builder'),
        slug: 'vivid-green-cyan',
        color: '#00d084',
    },
    {
        name: __('Pale cyan blue', 'funnel-builder'),
        slug: 'pale-cyan-blue',
        color: '#8ed1fc',
    },
    {
        name: __('Vivid cyan blue', 'funnel-builder'),
        slug: 'vivid-cyan-blue',
        color: '#0693e3',
    },
    {
        name: __('Vivid purple', 'funnel-builder'),
        slug: 'vivid-purple',
        color: '#9b51e0',
    },
];
export const textAlignSetting = [
    {
        id: 'left',
        icon: 'editor-alignleft',
        label: __('Left', 'funnel-builder'),
    },
    {
        id: 'center',
        icon: 'editor-aligncenter',
        label: __('Center', 'funnel-builder'),
    },
    {
        id: 'right',
        icon: 'editor-alignright',
        label: __('Right', 'funnel-builder'),
    },
    {
        id: 'justify',
        icon: 'editor-justify',
        label: __('Justify', 'funnel-builder'),
    },
];
export const textDecorationSetting = [
    {
        id: 'none',
        icon: 'editor-textcolor',
        label: __('None', 'funnel-builder'),
    },
    {
        id: 'line-through',
        icon: 'editor-strikethrough',
        label: __('Line Through', 'funnel-builder'),
    },
    {
        id: 'underline',
        icon: 'editor-underline',
        label: __('Underline', 'funnel-builder'),
    },
    {
        id: 'overline',
        icon: 'ellipsis',
        label: __('Overline', 'funnel-builder'),
    },
];

export const textTransformSetting = [
    {
        id: 'none',
        label: __('None', 'funnel-builder'),
    },
    {
        id: 'capitalize',
        label: __('Capitalize', 'funnel-builder'),
    },
    {
        id: 'uppercase',
        label: __('Uppercase', 'funnel-builder'),
    },
    {
        id: 'lowercase',
        label: __('Lowercase', 'funnel-builder'),
    },
];

const fontsarray =
    // eslint-disable-next-line camelcase
    typeof slingblocks !== 'undefined' &&
    slingblocks.slingblocks_g_font_names
        ? slingblocks.slingblocks_g_font_names.map((name) => {
            return {label: name, value: name, google: true};
        })
        : {};

export const FontFamilyList = [
    {
        type: 'group',
        label: __('Standard Fonts', 'funnel-builder'),
        options: typeof slingblocks !== 'undefined' && slingblocks && slingblocks.slingblockssystem_font_path ? slingblocks.slingblockssystem_font_path : [
            {
                label: 'System Default',
                value: '-apple-system,BlinkMacSystemFont,"Segoe UI",Roboto,Oxygen-Sans,Ubuntu,Cantarell,"Helvetica Neue",sans-serif, "Apple Color Emoji", "Segoe UI Emoji", "Segoe UI Symbol"',
                google: false,
            },
            {
                label: 'Arial, Helvetica, sans-serif',
                value: 'Arial, Helvetica, sans-serif',
                google: false,
            },
            {
                label: '"Arial Black", Gadget, sans-serif',
                value: '"Arial Black", Gadget, sans-serif',
                google: false,
            },
            {
                label: 'Helvetica, sans-serif',
                value: 'Helvetica, sans-serif',
                google: false,
            },
            {
                label: '"Comic Sans MS", cursive, sans-serif',
                value: '"Comic Sans MS", cursive, sans-serif',
                google: false,
            },
            {
                label: 'Impact, Charcoal, sans-serif',
                value: 'Impact, Charcoal, sans-serif',
                google: false,
            },
            {
                label: '"Lucida Sans Unicode", "Lucida Grande", sans-serif',
                value: '"Lucida Sans Unicode", "Lucida Grande", sans-serif',
                google: false,
            },
            {
                label: 'Tahoma, Geneva, sans-serif',
                value: 'Tahoma, Geneva, sans-serif',
                google: false,
            },
            {
                label: '"Trebuchet MS", Helvetica, sans-serif',
                value: '"Trebuchet MS", Helvetica, sans-serif',
                google: false,
            },
            {
                label: 'Verdana, Geneva, sans-serif',
                value: 'Verdana, Geneva, sans-serif',
                google: false,
            },
            {
                label: 'Georgia, serif',
                value: 'Georgia, serif',
                google: false,
            },
            {
                label: '"Palatino Linotype", "Book Antiqua", Palatino, serif',
                value: '"Palatino Linotype", "Book Antiqua", Palatino, serif',
                google: false,
            },
            {
                label: '"Times New Roman", Times, serif',
                value: '"Times New Roman", Times, serif',
                google: false,
            },
            {
                label: 'Courier, monospace',
                value: 'Courier, monospace',
                google: false,
            },
            {
                label: '"Lucida Console", Monaco, monospace',
                value: '"Lucida Console", Monaco, monospace',
                google: false,
            },
        ],
    },
    {
        type: 'group',
        label: __('Google Fonts', 'funnel-builder'),
        options: fontsarray,
    },
];
export const systemFonts = FontFamilyList[0].options.map(opt => opt.value);

export const FontWeightList = [
    {
        value: 'inherit',
        label: __('Inherit', 'funnel-builder'),
    },
    {value: '100', label: __('Thin 100', 'funnel-builder')},
    {
        value: '200',
        label: __('Extra-Light 200', 'funnel-builder'),
    },
    {value: '300', label: __('Light 300', 'funnel-builder')},
    {value: '400', label: __('Regular', 'funnel-builder')},
    {
        value: '500',
        label: __('Medium 500', 'funnel-builder'),
    },
    {
        value: '600',
        label: __('Semi-Bold 600', 'funnel-builder'),
    },
    {value: '700', label: __('Bold 700', 'funnel-builder')},
    {
        value: '800',
        label: __('Extra-Bold 800', 'funnel-builder'),
    },
    {
        value: '900',
        label: __('Ultra-Bold 900', 'funnel-builder'),
    },
];

export const faGetSVGIcon = (prefix, iconName) => {
    const icon = faGetIcon(prefix, iconName);
    if (icon) {
        return icon.html[0];
    }
    return '';
};

export const faGetIcon = (prefix, iconName) => {
    if (!window.FontAwesome) {
        return null;
    }
    return window.FontAwesome.icon({prefix, iconName});
};

export const faIsAPILoaded = () => {
    return !!window.FontAwesome;
};

export const faAPILoaded = () => {
    if (!window.FontAwesome) {
        return new Promise((resolve, reject) => {
            let timeoutCounter = 240;
            const interval = setInterval(() => {
                if (window.FontAwesome) {
                    clearInterval(interval);
                    resolve(true);
                } else if (timeoutCounter-- < 0) {
                    clearInterval(interval);
                    reject(false);
                }
            }, 250);
        });
    }
    return Promise.resolve(true);
};

export const faIconLoaded = (prefix, iconName) => {
    const icon = faGetIcon(prefix, iconName);
    if (!icon) {
        return new Promise((resolve, reject) => {
            let timeoutCounter = 240;
            const interval = setInterval(() => {
                const icon = faGetIcon(prefix, iconName);
                if (window.FontAwesome) {
                    clearInterval(interval);
                    resolve(icon);
                } else if (timeoutCounter-- < 0) {
                    clearInterval(interval);
                    reject(false);
                }
            }, 250);
        });
    }
    return Promise.resolve(icon);
};

export const getShapeSVG = (shape) => {
    const SVGS = {
        circle: SVGCircle,
        square: SVGSquare,
        blob1: SVGBlob1,
    };
    return !SVGS[shape] ? null : SVGS[shape];
};

export const DEFAULT_CHECK_SVG =
    '<svg data-prefix="fas" data-icon="check" class="svg-inline--fa fa-check fa-w-16" role="img" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 512 512"><path fill="currentColor" d="M173.898 439.404l-166.4-166.4c-9.997-9.997-9.997-26.206 0-36.204l36.203-36.204c9.997-9.998 26.207-9.998 36.204 0L192 312.69 432.095 72.596c9.997-9.997 26.207-9.997 36.204 0l36.203 36.204c9.997 9.997 9.997 26.206 0 36.204l-294.4 294.401c-9.998 9.997-26.207 9.997-36.204-.001z"></path></svg>';
export const DEFAULT_TICK_SVG =
    '<svg data-prefix="far" data-icon="check-circle" role="img" xmlns="http://www.w3.org/2000/svg" viewBox="0 0 512 512" > <path fill="currentColor" d="M256 8C119.033 8 8 119.033 8 256s111.033 248 248 248 248-111.033 248-248S392.967 8 256 8zm0 48c110.532 0 200 89.451 200 200 0 110.532-89.451 200-200 200-110.532 0-200-89.451-200-200 0-110.532 89.451-200 200-200m140.204 130.267l-22.536-22.718c-4.667-4.705-12.265-4.736-16.97-.068L215.346 303.697l-59.792-60.277c-4.667-4.705-12.265-4.736-16.97-.069l-22.719 22.536c-4.705 4.667-4.736 12.265-.068 16.971l90.781 91.516c4.667 4.705 12.265 4.736 16.97.068l172.589-171.204c4.704-4.668 4.734-12.266.067-16.971z" ></path> </svg>';

export const computeSpaceValue = (prop, data) => {
    const temp = lodash.cloneDeep(prop ? prop : {});
    Object.keys(temp).forEach(function (key, idx) {
        temp[key] = data[idx];
    });
    return temp;
};
const TABS = [
    {
        id: 'normal',
        label: __('Normal', 'funnel-builder'),
    },
    {
        id: 'hover',
        label: __('Hover', 'funnel-builder'),
    },
];

export const hoverTab = (screenType, setScreen, extraTab = [], customTab) => {
    const tabArr = customTab ? customTab : [...TABS, ...extraTab];
    return (
        <div className={classNames('bwf-responsive-tabs-list bwf-hover-tab')}>
            {tabArr.map((TData) => {
                return (
                    <Button
                        key={TData.id}
                        onClick={() => {
                            setScreen(TData.id);
                        }}
                        className={classNames('bwf-screen-tabs', {
                            'active-screen': screenType === TData.id,
                        })}
                    >
                        <Tooltip position="top center" text={TData.label}>
                          <span className={'bwf-device-brick'}>
                              {TData.label}
                          </span>
                        </Tooltip>
                    </Button>
                );
            })}
        </div>
    );
}

export const rgba2hex = (orig) => {
    let a,
        rgb = orig
            .replace(/\s/g, '')
            .match(/^rgba?\((\d+),(\d+),(\d+),?([^,\s)]+)?/i),
        alpha = ((rgb && rgb[4]) || '').trim(),
        hex = rgb
            ? (rgb[1] | (1 << 8)).toString(16).slice(1) +
            (rgb[2] | (1 << 8)).toString(16).slice(1) +
            (rgb[3] | (1 << 8)).toString(16).slice(1)
            : orig;

    if (alpha !== '') {
        a = alpha;
    } else {
        a = 1;
    }
    // multiply before convert to HEX
    a = ((a * 255) | (1 << 8)).toString(16).slice(1);
    hex = hex + a;
    return `#${hex}`;
};

export const DimensionToggle = (props) => {
    const {
        onChange = noop,
        initial = false,
    } = props;
    const [toggle, setToggle] = useState(initial);
    const handleToggle = () => {
        setToggle(!toggle)
        onChange(!toggle)
    }
    return (
        toggle
            ? <span className={classNames('bwf-dimension-toggle')} onClick={handleToggle} title={'Edit Indivisual'}>
                <IconSquare/>
            </span>
            : <span className={classNames('bwf-dimension-toggle')} onClick={handleToggle} title={'Edit All'}>
                <IconIndivisual/>
            </span>
    )
}

export const breakPoints = (props = {}) => {
    const {clientId} = props;
    return {
        'tablet': clientId ? '5000px' : '1024px',
        'mobile': clientId ? '5000px' : '767px',
    }
}

/**
 * Simple CSS minification.
 *
 * @see https://stackoverflow.com/questions/15411263/how-do-i-write-a-better-regexp-for-css-minification-in-php
 *
 * @param {string} css CSS to minify.
 * @param {boolean} important Add !important to all rules.
 *
 * @return {string} Minified CSS
 */
export const minifyCSS = (css, important = false) => {
    const minified = css.replace(/\/\*.*?\*\//g, '') // Comments.
        .replace(/\n\s*\n/g, '') // Comments.
        .replace(/[\n\r \t]/g, ' ') // Spaces.
        .replace(/ +/g, ' ') // Multi-spaces.
        .replace(/ ?([,:;{}]) ?/g, '$1') // Extra spaces.
        .replace(/[^\}\{]+\{\}/g, '') // Blank selectors.
        .replace(/[^\}\{]+\{\}/g, '') // Blank selectors. Repeat to catch empty media queries.
        .replace(/;}/g, '}') // Trailing semi-colon.
        .trim()

    if (!important) {
        return minified
    }

    return minified
        .replace(/\s?\!important/g, '') // Remove all !important
        .replace(/([;\}])/g, ' !important$1') // Add our own !important.
        .replace(/\} !important\}/g, '}}') // Ending of media queries "}}" get an added !important from the previous line, remove it.
        .trim();


}

export const hasBlockVisible = (selector = '', props, screen = 'desktop') => {
    if (!selector) {
        return '';
    }

    const {attributes: {vsdesk, vsmobile, vstablet}} = props;
    let visibilityStyle = '';
    switch (screen) {
        case 'desktop':
            if (true === vsdesk) {
                visibilityStyle += `${selector}{opacity:0.1};`;
            }
            break;
        case 'tablet':
            if (true === vstablet) {
                visibilityStyle += `${selector}{opacity:0.1};`;
            }
            break;
        case 'mobile':
            if (true === vsmobile) {
                visibilityStyle += `${selector}{opacity:0.1};`;
            }
            break;
    }
    if (visibilityStyle) {
        return (
            <style>{visibilityStyle}</style>
        )
    }
    return '';
}

/**
 * function to return string with capital letter.
 * @param {string} string the word string.
 * @returns {string} with capital letter.
 */
export const capitalizeFirstLetter = (string) => {
    return string.charAt(0).toUpperCase() + string.slice(1);
}

export const getCurrentScreen = () => {
    var deviceType = useSelect((select) => {
        const {
            __experimentalGetPreviewDeviceType = null,
        } = select('core/edit-post');
        return __experimentalGetPreviewDeviceType ? __experimentalGetPreviewDeviceType() : 'Desktop';
    }, []);
    deviceType = deviceType ? deviceType.toLowerCase() : 'desktop';
    return deviceType;
}

wp.domReady(function () {
    document.addEventListener('click', (e) => {
        if ('SELECT' === e.target.tagName && e.target.querySelector('[value="wftp-canvas.php"]')) {
            if ('wftp-canvas.php' === e.target.value) {
                document.body.classList.add('bwf-editor-width-canvas');
                document.body.classList.remove('bwf-editor-width-boxed');

            } else if ('wftp-boxed.php' === e.target.value) {
                document.body.classList.add('bwf-editor-width-boxed');
                document.body.classList.remove('bwf-editor-width-canvas');
            } else {
                document.body.classList.remove('bwf-editor-width-canvas');
                document.body.classList.remove('bwf-editor-width-boxed');
            }
        }
    });
});

export const WP_VERSION = window.hasOwnProperty( 'wfty_blocks' ) ? wfty_blocks.wp_version : '';

export const includeFontInIframe = (fonts, prevFont = {}) => {
    if(fonts && select('core/edit-post').__experimentalGetPreviewDeviceType() !== 'Desktop'){
        setTimeout(() => {
            const editorFrame = document.querySelector( 'iframe[name="editor-canvas"]' );
            if(editorFrame){
                const head = editorFrame.contentDocument.head;
                fonts.forEach((value) => {
                    if(!isEmpty(value)){
                        let link = document.createElement('link')
                        link.rel = 'stylesheet';
                        link.href = `https://fonts.googleapis.com/css?family=${value}`
                        head.appendChild(link)

                        if(fonts.length === 1){
                            if(prevFont.current){
                                let prevLink = editorFrame.contentDocument.querySelector(`link[href*="family=${prevFont.current}"]`);
                                if(prevLink){
                                    prevLink.remove();
                                }
                            }
                            prevFont.current = value
                        }
                    }
                })
            }
        }, 500)
    }
}